﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace Inet.Viewer.WinForms
{
    /// <summary>
    /// A modal dialog which shows a username text field and a password field to login. This class 
    /// implements IAuthenticator to handle authentication requests from network communication 
    /// objects (i.e. URLRenderData).
    /// </summary>
    public partial class AuthenticationDialog : Form, Inet.Viewer.Data.URLRenderData.IAuthenticator
    {
        private Form owner;
        private ManualResetEvent mre = new ManualResetEvent(false);
        private NetworkCredential credentials;
        private object waitLock = new object();
        
        /// <summary>
        /// Creates a new dialog with the specified owner.
        /// </summary>
        /// <param name="owner">the owener</param>
        public AuthenticationDialog(Form owner)
        {
            this.owner = owner;
            InitializeComponent();
        }

        /// <summary>
        /// Opens the dialogs and wait for the user's input.
        /// </summary>
        /// <param name="url">the URL to log in</param>
        /// <param name="message">the message of the server</param>
        /// <returns>a credential object with the user's input or null if the user had canceled the dialog</returns>
        public NetworkCredential Authenticate(string url, string message)
        {
            lock (waitLock)
            {
                owner.BeginInvoke(new Action(() => AuthenticateUI(url, message)));
                // wait until the dialog was closed, then return the user's input
                Monitor.Wait(waitLock);
                return credentials;
            }
        }

        /// <summary>
        /// Opens the dialog and informs other threads when the user closed the dialog.
        /// May be called from the UI thread only. 
        /// </summary>
        /// <param name="url">the URL to log in</param>
        /// <param name="message">the message of the server</param>
        private void AuthenticateUI(string url, string message)
        {
            if (Visible)
            {
                // if the dialog was already opened, do nothing here
                // no synchronization needed here since we are in the UI thread
                return;
            }
            tbMessage.Text = "Login required for " + url + "\r\n" + message;
            tbUser.Text = string.Empty;
            tbPassword.Text = string.Empty;
            ActiveControl = tbUser;
            if (ShowDialog(owner) == DialogResult.OK)
            {
                credentials = new NetworkCredential(tbUser.Text, tbPassword.Text);
            }
            else
            {
                credentials = null;
            }
            // inform all waiting threads
            lock (waitLock)
            {
                Monitor.PulseAll(waitLock);
            }
        }

        /// <summary>
        /// Called on keypresses in the textfield and password field
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event parameters</param>
        private void OnKeyPress(object sender, KeyPressEventArgs e)
        {
            if (e.KeyChar == (char)Keys.Return)
            {
                Close();
                DialogResult = DialogResult.OK;
            }
        }
 
    }
}
